# Universal Object Removal Script (removealluniversal.ps1) v1.1
# For more information, see: https://vswitchzero.com/2018/11/16/removing-all-universal-objects-using-powernsx-and-powershell-scripting/
# This script will remove all universal objects from an NSX manager. To be used only with Transit role managers.

# First, display a warning and ensure user is connected to the correct manager.
cls
echo " "
Write-Host -foregroundcolor yellow "This script is designed to remove all universal objects from an NSX manager in the 'Transit' role so that it can become a 'Standalone' manager. Please ensure that you are connected to the correct 'Transit' NSX Manager and associated vCenter Server in PowerCLI/PowerNSX before proceeding. Please do not proceed unless you have taken a backup of both the Primary and Transit NSX managers. No warranty or guarantees are provided with this script. Although it has been tested to function correctly, use it at your own risk."
echo " "
Write-Host -NoNewLine "Are you sure you have taken a backup and wish to proceed? [y/n]:"
$YNresponse = read-host
if ( $YNresponse -ne "y" ) {exit}

# Checking to ensure the connected NSX manager is in the Transit role
echo " "
echo "Checking to ensure the connected NSX Manager is in the 'Transit' role..."

Get-NsxManagerRole
If ((Get-NsxManagerRole).role -eq "TRANSIT") {
	echo " "
	echo "Confirmed role is 'Transit'. Proceeding..."} 
Else {
	echo " "
	echo "The currently connected NSX Manager is not in the 'Transit' role. Are you sure you are connected to the correct NSX Manager?"
	exit
}

# Getting counts of all universal objects in the environment:

echo " "
echo "Getting counts of all Universal Objects associated with this NSX Manager. This may take a minute..."

$countTZ = Get-NsxTransportZone | where{$_.isUniversal -eq "true"}|measure
$countSID = Get-NsxSegmentIdRange |where{$_.isUniversal -eq "true"}|measure
$countUDLR = Get-NsxLogicalRouter |where{$_.isUniversal -eq "true"}|measure
$countULS = Get-NsxLogicalSwitch | where{$_.isUniversal -eq "true"}|measure
$countFWSEC = Get-NsxFirewallSection | where{$_.managedBy -eq "universalroot-0"} |measure
$countFWSVG = Get-NsxServiceGroup -scopeid universalroot-0 |measure
$countFWSV = Get-NsxService -UniversalOnly |measure
$countFWSG = Get-NsxSecurityGroup -scopeid universalroot-0 |measure
$countIPS = Get-NsxIpSet -scopeid universalroot-0  |measure
$countUST = Get-NsxSecurityTag | where{$_.isUniversal -eq "true"}|measure

echo "Done."
echo " "
echo "Total count of all Universal Object types:"
echo " "

# Displays counts of all object types:

echo "Universal Transport Zones: $($countTZ.count)" 
echo "Universal Segment ID Ranges: $($countSID.count)"
echo "Universal Logical Routers: $($countUDLR.count)"
echo "Universal Logical Switches: $($countULS.count)"
echo "Universal Firewall Sections: $($countFWSEC.count)"
echo "Universal Firewall Service Groups: $($countFWSVG.count)"
echo "Universal Firewall Services: $($countFWSV.count)"
echo "Universal Firewall Security Groups: $($countFWSG.count)"
echo "Universal IP Sets: $($countIPS.count)"
echo "Universal Security Tags: $($countUST.count)"

# Confirm before proceeding

echo " "
Write-Host -NoNewLine "Remove all universal objects from this NSX Manager? [y/n]:"
$YNresponse = read-host
if ( $YNresponse -ne "y" ) {exit}

# Start removing all of the object types and report the number successfully removed.
echo "Removing all universal objects. This may take several minutes..."
echo " "

echo "Removing Universal Logical Routers..."
Get-NsxLogicalRouter |where{$_.isUniversal -eq "true"}| Remove-NsxLogicalRouter -Confirm:$false
$afterUDLR = Get-NsxLogicalRouter |where{$_.isUniversal -eq "true"}|measure
$removedUDLR = $countUDLR.count - $afterUDLR.count
echo "$($removedUDLR) object(s) successfully removed."

echo "Removing Universal Logical Switches..."
Get-NsxLogicalSwitch | where{$_.isUniversal -eq "true"}| Remove-NsxLogicalSwitch -Confirm:$false
$afterULS = Get-NsxLogicalSwitch | where{$_.isUniversal -eq "true"}|measure
$removedULS = $countULS.count - $afterULS.count
echo "$($removedULS) object(s) successfully removed."

echo "Removing Universal Firewall Sections..."
Get-NsxFirewallSection | ? {$_.managedBy -eq "universalroot-0"} | Remove-NsxFirewallSection -confirm:$false -force
$afterFWSEC = Get-NsxFirewallSection | where{$_.managedBy -eq "universalroot-0"} |measure
$removedFWSEC = $countFWSEC.count - $afterFWSEC.count
echo "$($removedFWSEC) object(s) successfully removed."

echo "Removing Universal Firewall Service Groups..."
Get-NsxServiceGroup -scopeid universalroot-0 | Remove-nsxservicegroup -confirm:$false -ErrorAction SilentlyContinue
$afterFWSVG = Get-NsxServiceGroup -scopeid universalroot-0 |measure
$removedFWSVG = $countFWSVG.count - $afterFWSVG.count
echo "$($removedFWSVG) object(s) successfully removed."
echo "$($afterFWSVG.count) object(s) not removed. These are likely read-only service groups and can be ignored."

echo "Removing Universal Firewall Services..."
Get-NsxService -UniversalOnly | Remove-NsxService -confirm:$false -ErrorAction SilentlyContinue
$afterFWSV = Get-NsxService -UniversalOnly |measure
$removedFWSV = $countFWSV.count - $afterFWSV.count
echo "$($removedFWSV) object(s) successfully removed."
echo "$($afterFWSV.count) object(s) not removed. These are likely read-only service groups and can be ignored."

echo "Removing Universal Firewall Security Groups..."
Get-NsxSecurityGroup -scopeid universalroot-0 | Remove-NsxSecurityGroup -confirm:$false
$afterFWSG = Get-NsxSecurityGroup -scopeid universalroot-0 |measure
$removedFWSG = $countFWSG.count - $afterFWSG.count
echo "$($removedFWSG) object(s) successfully removed."

echo "Removing Universal IP Sets..."
Get-NsxIpSet -scopeid universalroot-0 | Remove-NsxIpSet -confirm:$false
$afterIPS = Get-NsxIpSet -scopeid universalroot-0  |measure
$removedIPS = $countIPS.count - $afterIPS.count
echo "$($removedIPS) object(s) successfully removed."

echo "Removing Universal Security Tags..."
Get-NsxSecurityTag | where{$_.isUniversal -eq "true"}|Remove-NsxSecurityTag -Confirm:$false
$afterUST = Get-NsxSecurityTag | where{$_.isUniversal -eq "true"}|measure
$removedUST = $countUST.count - $afterUST.count
echo "$($removedUST) object(s) successfully removed."

echo "Removing Universal Transport Zones..."
Get-NsxTransportZone | where{$_.isUniversal -eq "true"}| Remove-NsxTransportZone -Confirm:$false
$afterTZ = Get-NsxTransportZone | where{$_.isUniversal -eq "true"}|measure
$removedTZ = $countTZ.count - $afterTZ.count
echo "$($removedTZ) object(s) successfully removed."

echo "Removing Universal Segment ID Ranges..."
Get-NsxSegmentIdRange |where{$_.isUniversal -eq "true"}|Remove-NsxSegmentIdRange -Confirm:$false
$afterSID = Get-NsxSegmentIdRange |where{$_.isUniversal -eq "true"}|measure
$removedSID = $countSID.count - $afterSID.count
echo "$($removedSID) object(s) successfully removed."

#Finish up
echo " "
echo "Finished!"
echo " "
echo "Total object counts after removal attempt:"
echo " "
echo "Universal Transport Zones: $($afterTZ.count)" 
echo "Universal Segment ID Ranges: $($afterSID.count)"
echo "Universal Logical Routers: $($afterUDLR.count)"
echo "Universal Logical Switches: $($afterULS.count)"
echo "Universal Firewall Sections: $($afterFWSEC.count)"
echo "Universal Firewall Service Groups: $($afterFWSVG.count)"
echo "Universal Firewall Services: $($afterFWSV.count)"
echo "Universal Firewall Security Groups: $($afterFWSG.count)"
echo "Universal IP Sets: $($afterIPS.count)"
echo "Universal Security Tags: $($afterUST.count)"
echo " "
echo "Note: It is expected to see non-zero counts for both Universal Service Groups and Services as there are internal read-only objects that can't be removed. Only user-created objects will be removed using this script. If all other counts are zero, you can now attempt to change the NSX Manager's role from Transit to Standalone."


